<?php

/**
 * Classe de configuração do Plugin Portal com suporte a múltiplas URLs
 */

if (!defined('GLPI_ROOT')) {
   die("Sorry. You can't access this file directly");
}

class PluginWbiframeConfig {
   
   /**
    * Retorna a URL padrão configurada do portal
    */
   public static function getDefaultPortalUrl() {
      $config = Config::getConfigurationValues('plugin:wbiframe');
      $urls = self::getPortalUrls();
      return !empty($urls) ? $urls[0]['url'] : 'https://wbuarque.com.br';
   }
   
   /**
    * Retorna todas as URLs configuradas
    */
   public static function getPortalUrls() {
   // Recupera configuração bruta
   $config = Config::getConfigurationValues('plugin:wbiframe');
   $urls = [];

   if (isset($config['portal_urls'])) {
      if (is_string($config['portal_urls'])) {
         $decoded = json_decode($config['portal_urls'], true);
         if (is_array($decoded)) {
            $urls = $decoded;
         }
      } elseif (is_array($config['portal_urls'])) {
         // Compatibilidade com versões antigas que gravavam array direto
         $urls = $config['portal_urls'];
      }
   }

   // Garante estrutura padrão
   $clean = [];
   foreach ($urls as $url_data) {
      if (!isset($url_data['url'])) {
         continue;
      }
      $clean[] = [
         'url'    => $url_data['url'],
         'name'   => $url_data['name']   ?? $url_data['url'],
         'active' => (bool)($url_data['active'] ?? false),
      ];
   }

   // Se não houver URLs configuradas, define WBuarque como padrão
   if (empty($clean)) {
      $clean[] = [
         'url'    => 'https://wbuarque.com.br',
         'name'   => 'WBuarque',
         'active' => true,
      ];
   }

   // Se não houver URLs, retorna array vazio
   return $clean;
}

/**
 * Define a lista de URLs do portal (armazenadas em JSON)
 */
public static function setPortalUrls($urls) {
   $clean_urls = [];
   foreach ($urls as $url_data) {
      if (!isset($url_data['url'])) {
         continue;
      }
      $clean_urls[] = [
         'url'    => $url_data['url'],
         'name'   => $url_data['name']   ?? $url_data['url'],
         'active' => (bool)($url_data['active'] ?? false),
      ];
   }

   return Config::setConfigurationValues('plugin:wbiframe', [
      'portal_urls' => json_encode($clean_urls, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
   ]);
}

/**

    * Adiciona uma nova URL
    */
   public static function addPortalUrl($url, $name = '') {
      $urls = self::getPortalUrls();
      
      // Verificar se já existe
      foreach ($urls as $existing) {
         if ($existing['url'] === $url) {
            return false; // URL já existe
         }
      }
      
      // Adicionar nova URL
      $urls[] = [
         'url' => $url,
         'name' => !empty($name) ? $name : self::getUrlDomain($url),
         'active' => false
      ];
      
      return self::setPortalUrls($urls);
   }
   
   /**
    * Remove uma URL
    */
   public static function removePortalUrl($url) {
      $urls = self::getPortalUrls();
      $filtered_urls = [];
      
      foreach ($urls as $url_data) {
         if ($url_data['url'] !== $url) {
            $filtered_urls[] = $url_data;
         }
      }
      
      return self::setPortalUrls($filtered_urls);
   }
   
   /**
    * Define URL como ativa
    */
   public static function setActiveUrl($url) {
      $urls = self::getPortalUrls();
      
      foreach ($urls as &$url_data) {
         $url_data['active'] = ($url_data['url'] === $url);
      }
      
      return self::setPortalUrls($urls);
   }
   
   /**
    * Verifica se a URL é válida
    */
   public static function isValidUrl($url) {
      return filter_var($url, FILTER_VALIDATE_URL) !== false;
   }
   
   /**
    * Extrai o domínio da URL para usar como nome padrão
    */
   public static function getUrlDomain($url) {
      $parsed = parse_url($url);
      return $parsed['host'] ?? 'Portal';
   }
   
   

   /**
    * Obtém URL ativa por usuário
    */
   public static function getUserActivePortalUrl($users_id = 0) {
      if (empty($users_id)) {
         if (class_exists('Session')) {
            $users_id = Session::getLoginUserID();
         }
      }

      if (empty($users_id)) {
         // Se não houver usuário logado, usa URL ativa global
         return self::getActivePortalUrl();
      }

      // Configuração específica do usuário
      $user_config = Config::getConfigurationValues('plugin:wbiframe_user_' . (int)$users_id);

      if (!empty($user_config['active_url'])) {
         return $user_config['active_url'];
      }

      // Fallback para URL ativa global
      return self::getActivePortalUrl();
   }

   /**
    * Define URL ativa por usuário
    */
   public static function setUserActiveUrl($url, $users_id = 0) {
      if (empty($users_id)) {
         if (class_exists('Session')) {
            $users_id = Session::getLoginUserID();
         }
      }

      if (empty($users_id)) {
         return false;
      }

      // Valida se a URL existe na lista configurada
      $urls = self::getPortalUrls();
      $valid = false;

      foreach ($urls as $url_data) {
         if ($url_data['url'] === $url) {
            $valid = true;
            break;
         }
      }

      if (!$valid) {
         return false;
      }

      return Config::setConfigurationValues(
         'plugin:wbiframe_user_' . (int)$users_id,
         [
            'active_url' => $url
         ]
      );
   }

/**
    * Obtém URL ativa
    */
   public static function getActivePortalUrl() {
      $urls = self::getPortalUrls();
      foreach ($urls as $url_data) {
         if ($url_data['active']) {
            return $url_data['url'];
         }
      }
      return !empty($urls) ? $urls[0]['url'] : 'https://wbuarque.com.br';
   }
}