<?php
/**
 * Portal -> WBStore notification receiver
 * Accepts:
 *  - POST JSON: { token, instance_id, license_key, title, message, level, url }
 *  - x-www-form-urlencoded
 *  - token via header: X-WBSTORE-Token (any casing)
 *  - token via Authorization: Bearer <token>
 */

include("../../../inc/includes.php");

use GlpiPlugin\Wbstore\Config;
use GlpiPlugin\Wbstore\Notifications;

header('Content-Type: application/json; charset=utf-8');

// ---- Read raw body once (avoid double-read of php://input) ----
$raw = (string)file_get_contents('php://input');
$json = [];
if (trim($raw) !== '') {
   $tmp = json_decode($raw, true);
   if (is_array($tmp)) {
      $json = $tmp;
   }
}

// ---- Token extraction (robust, case-insensitive) ----
$token = (string)($_GET['token'] ?? ($_POST['token'] ?? ($json['token'] ?? '')));

$headers = function_exists('getallheaders') ? (array)getallheaders() : [];
foreach ($headers as $k => $v) {
   if (strtolower((string)$k) === 'x-wbstore-token') {
      $token = $token !== '' ? $token : (string)$v;
      break;
   }
}

if ($token === '' && !empty($_SERVER['HTTP_X_WBSTORE_TOKEN'])) {
   $token = (string)$_SERVER['HTTP_X_WBSTORE_TOKEN'];
}

// Some servers only pass Authorization via REDIRECT_HTTP_AUTHORIZATION
$auth = (string)($_SERVER['HTTP_AUTHORIZATION'] ?? ($_SERVER['REDIRECT_HTTP_AUTHORIZATION'] ?? ''));
if ($token === '' && $auth !== '') {
   if (preg_match('/^Bearer\s+(.*)$/i', trim($auth), $m)) {
      $token = trim((string)$m[1]);
   }
}

$expected = (string)Config::get('portal_token', '');

// If portal_token was not generated/saved for some reason, bootstrap from first valid request.
if ($expected === '') {
   if (trim($token) === '') {
      http_response_code(403);
      echo json_encode(['ok' => false, 'error' => 'missing_token']);
      exit;
   }
   Config::set(['portal_token' => $token]);
   $expected = $token;
}

if (!hash_equals($expected, (string)$token)) {
   http_response_code(403);
   echo json_encode(['ok' => false, 'error' => 'invalid_token']);
   exit;
}

// ---- Payload ----
$data = [];
if (!empty($json)) {
   $data = $json;
} else if (!empty($_POST)) {
   $data = $_POST;
}

$iid = (string)($data['instance_id'] ?? '');
$lk  = (string)($data['license_key'] ?? '');

if ($iid === '' || $lk === '') {
   http_response_code(400);
   echo json_encode(['ok' => false, 'error' => 'missing_instance_or_license']);
   exit;
}

// Only allow per configured instance/license
if ($iid !== (string)Config::get('instance_id', '') || $lk !== (string)Config::get('license_key', '')) {
   http_response_code(404);
   echo json_encode(['ok' => false, 'error' => 'not_found']);
   exit;
}

$title   = (string)($data['title'] ?? 'Aviso');
$message = (string)($data['message'] ?? ($data['html'] ?? ''));
$level   = (string)($data['level'] ?? 'info');
$url     = (string)($data['url'] ?? ($data['action_url'] ?? ''));

Notifications::create([
   'instance_id' => $iid,
   'license_key' => $lk,
   'title'       => $title,
   'message'     => $message,
   'level'       => $level,
   'url'         => $url,
   'is_read'     => 0,
]);

echo json_encode(['ok' => true]);
