<?php
if (!defined('GLPI_ROOT')) {
   die("Sorry. You can't access this file directly");
}

use GlpiPlugin\Wbstore\Guard;
use GlpiPlugin\Wbstore\License;
use GlpiPlugin\Wbstore\Utils;

/**
 * CronTasks do WBSTORE
 */
class PluginWbstoreCron extends CommonDBTM {

   static function cronInfo($name) {
      switch ($name) {
         case 'WbstoreSync':
            return [
               'description' => __('WBStore: sincronizar licença/cache (Portal)'),
               'parameter'   => ''
            ];
      }
      return [];
   }

   /**
    * Executa sincronização online (licença + cache de pacotes).
    * Deve rodar apenas via cron do GLPI (modo externo).
    *
    * @param CronTask $task
    * @return int 0/1
    */
      /**
    * Garante que a tarefa apareça em Configurar > Ações automáticas.
    * Chamado apenas em telas administrativas (catálogo/config), nunca no INIT dos plugins.
    */
   static function ensureRegistered(): void {
      try {
         if (!class_exists('CronTask') || !method_exists('CronTask', 'Register')) {
            return;
         }
         // Throttle: tenta registrar no máximo 1x por dia
         $now = time();
         $last = (int)\GlpiPlugin\Wbstore\Config::get('cron_registered_at', 0);
         if ($last > 0 && ($now - $last) < 86400) {
            return;
         }
         \GlpiPlugin\Wbstore\Config::set(['cron_registered_at' => $now]);
         // Registra como MODE_EXTERNAL; a janela 01:00-04:00 é controlada no método cron.
         \CronTask::Register('PluginWbstoreCron', 'WbstoreSync', DAY_TIMESTAMP, [
            'mode'    => \CronTask::MODE_EXTERNAL,
            'comment' => 'WBStore: sincroniza licença/pacotes (cache local)'
         ]);
      } catch (Throwable $e) {
         // ignore
      }
   }

   static function cronWbstoreSync($task = null) {
      // Janela de execução: 01:00-04:00 (evita pico) - apenas no modo CLI (cron externo)
      $isCli = (PHP_SAPI === 'cli');
      $h = (int)date('G');
      if ($isCli && ($h < 1 || $h > 4)) {
         if ($task instanceof CronTask) {
            $task->setVolume(0);
         }
         return 0;
      }

      // Gera token local do portal (não vaza segredo)
      Utils::ensurePortalToken();

      $okLicense = false;
      $okCache   = false;

      try {
         $okLicense = License::isValid(true);
      } catch (Throwable $e) {
         $okLicense = false;
      }

      try {
         $okCache = Guard::refreshPackagesCache(true);
      } catch (Throwable $e) {
         $okCache = false;
      }

      if ($task instanceof CronTask) {
         $task->setVolume(($okLicense && $okCache) ? 1 : 0);
         if (!$okLicense || !$okCache) {
            $task->log('WBStoreSync: falha ao sincronizar (ver logs do portal/token/config).');
         } else {
            $task->log('WBStoreSync: sincronização OK.');
         }
      }

      return ($okLicense && $okCache) ? 1 : 0;
   }
}
